<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use App\Models\User;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): Response
    {
        return Inertia::render('Profile/Edit', [
            'mustVerifyEmail' => $request->user() instanceof MustVerifyEmail,
            'status' => session('status'),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {

        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('success', "Profile updated successfully");
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validate([
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/')->with('success', "Account deleted successfully");;
    }

    public function file(Request $request, $id): RedirectResponse
    {
        $user = User::query()->findOrFail($id);

        if ($request->avatar) {
            $image = explode('base64,', $request->avatar);
            $image = end($image);
            $image = str_replace(' ', '+', $image);
            $file = $user->name . uniqid() . '.jpg';

            $result = Storage::disk('public')->put("uploads/" . $file, base64_decode($image));
            if (!$result) {
                return Redirect::back()->with('error', 'Image upload failed, please try again');
            } else {
                $avatarName = $file;
            }
        } else {
            $avatarName = $user->avatar;
        }

        if ($request->cover) {
            $image = explode('base64,', $request->cover);
            $image = end($image);
            $image = str_replace(' ', '+', $image);
            $file = $user->name."bg". uniqid() . '.jpg';

            $result = Storage::disk('public')->put("uploads/" . $file, base64_decode($image));
            if (!$result) {
                return Redirect::back()->with('error', 'Image upload failed, please try again');
            } else {
                $coverName = $file;
            }
        } else {
            $coverName = $user->cover;
        }


        User::where('id', $id)->update([
            'avatar' => $avatarName,
            'cover' => $coverName
        ]);

        return Redirect::route('profile.edit')->with('success', "Profile updated successfully");
    }
}
