<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreMediaRequest;
use App\Models\Media;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        if ($request->search) {
            $user = User::query()
                ->where('name', 'like', $request->search . '%')
                ->orWhere('email', 'like', $request->search . '%')
                ->latest()
                ->paginate(100);
        } else {
            $user = User::query()->latest()->paginate(15);
        }

        return inertia('Users/Index', [
            'users' => $user,
        ]);
    }

    /**
     * Display the registration view.
     */
    public function create()
    {
        return inertia('Users/Create');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|lowercase|email|max:255|unique:' . User::class,
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'phone' => ['required', 'regex:/^\+\d{1,3}\d{9}$/'],
            'role' => ['required', 'string', 'max:255'],
        ]);
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
        ]);
        return redirect()->route('users.index')->with('success','Data added successfully!');;
    }

    public function edit(User $user)
    {
        return inertia("Users/Edit", [
            'user' => $user,
        ]);
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|string|lowercase|email|max:255|unique:users,email,' . $user->id,
            'password' => ['sometimes', 'confirmed'],
            'phone' => ['sometimes', 'regex:/^\+\d{1,3}\d{9}$/'],
        ]);

        if ($request->password) {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'role' => $request->role,
                'password' => Hash::make($request->password),
            ]);
        } else {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'role' => $request->role,
                'phone' => $request->phone,
            ]);
        }

        return redirect()->route('users.index')->with('success','Data updated successfully!');;
    }


    public function destroy(User $user): RedirectResponse
    {
        DB::transaction(function () use ($user) {
            Storage::disk('public')->delete("uploads/" . $user->avatar);
            Storage::disk('public')->delete("uploads/" . $user->cover);
            $user->delete();
        });
        return Redirect::back()->with('success','Data deleted successfully!');;
    }


    public function show(User $user)
    {
        return inertia('Users/Show', [
            'user' => $user,
            'medias' => Media::query()->where('user_id', $user->id)->get(),
        ]);
    }

    public function createUserLinks($id)
    {
        return inertia('Users/CreateUserLinks', [
            'userId' => $id
        ]);
    }

    public function storeUserLinks(StoreMediaRequest $request)
    {
        if ($request->name == 'whatsapp') {
            $url = 'https://wa.me/' . (int)$request->url;
        } else if ($request->name == 'telegram') {
            $url = 'https://t.me/' . $request->url;
        } else if ($request->name == 'messenger') {
            $url = 'http://m.me/' . $request->url;
        } else {
            $url = $request->url;
        }

        Media::create([
            'name' => $request->name,
            'title' => $request->title,
            'url' => $url,
            'user_id' => $request->userId,
        ]);

        return to_route('users.show', $request->userId)->with('success','Link added successfully!');;
    }

    public function destroyeUserLinks($id)
    {
        $media = Media::findOrFail($id);
        $media->delete();
        return Redirect::back()->with('success','Link removed successfully!');;
    }

    public function updateUserLinks(StoreMediaRequest $request, $id)
    {
        if ($request->name == 'whatsapp') {
            $url = 'https://wa.me/' . (int)$request->url;
        } else if ($request->name == 'telegram') {
            $url = 'https://t.me/' . $request->url;
        } else if ($request->name == 'messenger') {
            $url = 'http://m.me/' . $request->url;
        } else {
            $url = $request->url;
        }

        $media = Media::findOrFail($id);

        $media->update([
            'name' => $request->name,
            'title' => $request->title,
            'url' => $url,
            'user_id' => $request->userId,
        ]);

        return to_route('users.show', $request->userId)->with('success','Link updated successfully!');;
    }


    public function updateMediaActiveFeature(Request $request, $id)
    {
        $user = User::query()->findOrFail($id);

        $user->media_active = $request->media_active;
        $user->save();

        return Redirect::back()->with('success','Status updated successfully!');;
    }

    public function updateProductActiveFeature(Request $request, $id)
    {
        $user = User::query()->findOrFail($id);

        $user->product_active = $request->product_active;
        $user->save();

        return Redirect::back()->with('success','Status updated successfully!');;
    }
}
