<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Http\Requests\StoreProductRequest;
use App\Http\Requests\UpdateProductRequest;
use App\Models\Category;
use App\Models\Store;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        if ($request->search) {
            $products = Product::query()
                ->with(['category'])
                ->where('user_id', Auth::user()->id)
                ->where(function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->search . '%')
                        ->orWhereHas('category', function ($query) use ($request) {
                            $query->where('name', 'like', '%' . $request->search . '%');
                        });
                })
                ->latest()
                ->paginate(15)
                ->withQueryString();
        } else {
            $products = Product::query()
                ->with(['category'])
                ->where('user_id', Auth::user()->id)
                ->latest()
                ->paginate(15);
        }


        return inertia('Product/User/Index', [
            'products' => $products,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return inertia('Product/User/Create', [
            'categories' => Category::where('user_id', Auth::user()->id)->get()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreProductRequest $request)
    {

        $store = Store::query()->where('user_id', Auth::user()->id)->first();
        if (is_null($store)) {
            return Redirect::back()->with('error', 'Please create your store first, so you can add products to your store');
        }

        $productCount = Product::query()->where('store_id', $store->id)->count();

        if ($productCount >= $store->plan) {
            return Redirect::back()->with('error', 'Your have reached the plan, please upgrade your plan so you can add more products');
        }


        $imageNames = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $fileName = $image->hashName();
                $contents = file_get_contents($image);
                Storage::disk('public')->put("products/" . $fileName, $contents);
                $imageNames[] = $fileName;
            }
        }

        Product::create([
            'category_id' => $request->validated('category_id'),
            'name' => $request->validated('name'),
            'description' => $request->validated('description'),
            'price' => $request->validated('price') ?? 0,
            'user_id' => Auth::user()->id,
            'store_id' => Auth::user()->store->id,
            'images' => $imageNames,
        ]);
        return to_route('products.index')->with('success', 'Product Added successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        return inertia('Product/User/Edit', [
            'product' => $product,
            'categories' => Category::where('user_id', Auth::user()->id)->get()
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateProductRequest $request, Product $product)
    {
        $store = Store::query()->where('user_id', Auth::user()->id)->first();
        $productCount = Product::query()->where('store_id', $store->id)->count();

        if ($productCount >= $store->plan) {
            return Redirect::back()->with('error', 'Your have reached the plan, please upgrade your plan so you can add more products');
        }

        $imageNames = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $fileName = $image->hashName();
                $contents = file_get_contents($image);
                Storage::disk('public')->delete("products/" . $product->images);
                Storage::disk('public')->put("products/" . $fileName, $contents);
                $imageNames[] = $fileName;
            }
        }

        if ($request->hasFile('images')) {
            $product->update([
                'category_id' => $request->validated('category_id'),
                'name' => $request->validated('name'),
                'description' => $request->validated('description'),
                'price' => $request->validated('price') ?? 0,
                'user_id' => Auth::user()->id,
                'store_id' => Auth::user()->store->id,
                'images' => $imageNames,
            ]);
        } else {
            $product->update([
                'category_id' => $request->validated('category_id'),
                'name' => $request->validated('name'),
                'description' => $request->validated('description'),
                'price' => $request->validated('price') ?? 0,
                'user_id' => Auth::user()->id,
                'store_id' => Auth::user()->store->id,
            ]);
        }


        return to_route('products.index')->with('success', 'Product updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        if ($product->images) {
            foreach ($product->images as $image) {
                Storage::disk('public')->delete("products/" . $image);
            }
        }

        $product->delete();
        return Redirect::back()->with('success', 'Product deleted successfully.');
    }
}
