import { FormEventHandler, useState } from "react";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import PrimaryButton from "@/Components/PrimaryButton";
import TextInput from "@/Components/TextInput";
import { Head, Link, router, usePage } from "@inertiajs/react";
import "react-phone-number-input/style.css";
import { ArrowLeft, CircleAlertIcon } from "lucide-react";
import Textarea from "@/Components/Textarea";
import FileDropzone from "../Partials/FileDropzone";
import { Category, Product } from "@/types/custom";

export default function Edit({
    product,
    categories,
}: {
    product: Product;
    categories: Category[];
}) {
    const { errors } = usePage().props;

    const [category_id, setCategoryId] = useState(product.category_id);
    const [name, setName] = useState(product.name);
    const [description, setDescription] = useState(product.description);
    const [price, setPrice] = useState(product.price);
    const [files, setFiles] = useState([]);
    const [errorMessage, setErrorMessage] = useState("");

    const submit: FormEventHandler = (e) => {
        e.preventDefault();
        setErrorMessage("");

        if (files.length > 3) {
            return setErrorMessage("More than 3 images are not valid!");
        }

        router.post(
            route("products.update", product.id),
            {
                category_id,
                name,
                description,
                price,
                images: files.map(
                    (file) =>
                        //@ts-ignore
                        file.file
                ),
            },
            {
                forceFormData: true,
            }
        );
    };

    return (
        <AuthenticatedLayout>
            <Head title="Edit Product" />

            {errorMessage && (
                <div className="flex justify-center items-center mt-5 -mb-10">
                    <p className="p-2 flex gap-2 rounded-sm mb-4 bg-red-600 text-white">
                        <CircleAlertIcon /> {errorMessage}
                    </p>
                </div>
            )}
            <div className="flex justify-center items-center mt-10">
                <form
                    onSubmit={submit}
                    className="w-96 bg-white p-7 rounded-lg"
                >
                    <Link href={route("products.index")}>
                        <ArrowLeft size={16} className="mb-3" />
                    </Link>
                    <div>
                        <InputLabel
                            htmlFor="category_id"
                            value="Category Name"
                        />
                        <select
                            name="category_id"
                            id="category_id"
                            value={category_id}
                            className="mt-1 block w-full rounded-md border-gray-200"
                            onChange={(e) =>
                                setCategoryId(e.target.value as any)
                            }
                        >
                            <option value="">--</option>
                            {categories.map((category) => (
                                <option value={category.id}>
                                    {category.name}
                                </option>
                            ))}
                        </select>
                        <InputError
                            message={errors.category_id}
                            className="mt-2"
                        />
                    </div>
                    <div>
                        <InputLabel htmlFor="name" value="Product Name" />
                        <TextInput
                            id="name"
                            name="name"
                            value={name}
                            className="mt-1 block w-full"
                            autoComplete="name"
                            isFocused={true}
                            onChange={(e) => setName(e.target.value)}
                            required
                        />

                        <InputError message={errors.name} className="mt-2" />
                    </div>
                    <div className="mt-2">
                        <InputLabel
                            htmlFor="description"
                            value="Product Description"
                        />
                        <Textarea
                            id="description"
                            name="description"
                            value={description}
                            className="mt-1 block w-full"
                            autoComplete="description"
                            isFocused={true}
                            onChange={(e: any) =>
                                setDescription(e.target.value)
                            }
                        />

                        <InputError
                            message={errors.description}
                            className="mt-2"
                        />
                    </div>
                    <div className="mt-2">
                        <InputLabel htmlFor="price" value="Product Price" />
                        <TextInput
                            id="price"
                            name="price"
                            value={price}
                            type="number"
                            className="mt-1 block w-full"
                            autoComplete="price"
                            isFocused={true}
                            onChange={(e) => setPrice(e.target.value as any)}
                        />

                        <InputError message={errors.price} className="mt-2" />
                    </div>
                    <div className="mt-2">
                        <InputLabel htmlFor="images" value="Product Images" />
                        <FileDropzone files={files} setFiles={setFiles} />
                        <InputError message={errors.images} className="mt-2" />
                    </div>

                    <div className="flex items-center justify-end mt-4">
                        <PrimaryButton type="submit" className="ms-4">
                            Save
                        </PrimaryButton>
                    </div>
                </form>
            </div>
        </AuthenticatedLayout>
    );
}
